import { Request, Response } from 'express';
import { adminService } from './service';

function isSuperAdmin(admin: any) {
  return admin?.role === 'super_admin';
}
function hasPermission(admin: any, perm: string) {
  return isSuperAdmin(admin) || (admin?.permissions || []).includes(perm);
}

export const adminController = {
  // GET /admin/users
  async listAdmins(req: any, res: Response) {
    const admin = req.admin;
    if (!isSuperAdmin(admin)) return res.status(403).json({ error: 'Super admin only' });
    const { data, error } = await adminService.listAdmins();
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // POST /admin/users
  async createAdmin(req: any, res: Response) {
    const admin = req.admin;
    if (!isSuperAdmin(admin)) return res.status(403).json({ error: 'Super admin only' });
    const { data, error } = await adminService.createAdmin(req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.status(201).json({ success: true, data });
  },

  // PATCH /admin/users/:id/permissions
  async updateAdminPermissions(req: any, res: Response) {
    const admin = req.admin;
    if (!isSuperAdmin(admin)) return res.status(403).json({ error: 'Super admin only' });
    const { id } = req.params;
    const { data, error } = await adminService.updateAdminPermissions(id, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // GET /admin/settings
  async getSettings(req: any, res: Response) {
    const admin = req.admin;
    if (!hasPermission(admin, 'edit_settings')) return res.status(403).json({ error: 'Permission denied' });
    const { data, error } = await adminService.getSettings();
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // PATCH /admin/settings
  async updateSettings(req: any, res: Response) {
    const admin = req.admin;
    if (!hasPermission(admin, 'edit_settings')) return res.status(403).json({ error: 'Permission denied' });
    const { data, error } = await adminService.updateSettings(req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // GET /admin/logs
  async getLogs(req: any, res: Response) {
    const admin = req.admin;
    if (!isSuperAdmin(admin)) return res.status(403).json({ error: 'Super admin only' });
    const { data, error } = await adminService.getLogs();
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // PATCH /admin/feature-flags
  async updateFeatureFlags(req: any, res: Response) {
    const admin = req.admin;
    if (!hasPermission(admin, 'edit_settings')) return res.status(403).json({ error: 'Permission denied' });
    const { data, error } = await adminService.updateFeatureFlags(req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // PATCH /admin/kyc/:id
  async approveKYCOrPricing(req: any, res: Response) {
    const admin = req.admin;
    if (!hasPermission(admin, 'approve_kyc')) return res.status(403).json({ error: 'Permission denied' });
    const { id } = req.params;
    const { status, admin_notes } = req.body;
    const { data, error } = await adminService.approveKYCOrPricing(id, status, admin_notes);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // POST /admin/impersonate
  async impersonate(req: any, res: Response) {
    const admin = req.admin;
    if (!isSuperAdmin(admin)) return res.status(403).json({ error: 'Super admin only' });
    const { data, error } = await adminService.impersonate(admin, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // GET /admin/audit-trail
  async getAuditTrail(req: any, res: Response) {
    const admin = req.admin;
    if (!isSuperAdmin(admin)) return res.status(403).json({ error: 'Super admin only' });
    const { data, error } = await adminService.getAuditTrail();
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // GET /admin/support-tickets
  async getSupportTickets(req: any, res: Response) {
    const admin = req.admin;
    if (!hasPermission(admin, 'resolve_tickets')) return res.status(403).json({ error: 'Permission denied' });
    const { data, error } = await adminService.getSupportTickets();
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // PATCH /admin/support-tickets/:id
  async updateSupportTicket(req: any, res: Response) {
    const admin = req.admin;
    if (!hasPermission(admin, 'resolve_tickets')) return res.status(403).json({ error: 'Permission denied' });
    const { id } = req.params;
    const { data, error } = await adminService.updateSupportTicket(id, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // GET /admin/notifications
  async getAdminNotifications(req: any, res: Response) {
    const admin = req.admin;
    const { data, error } = await adminService.getAdminNotifications(admin.id);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  }
}; 